﻿
intellisense.enableMicrosoftAjaxLog = function (logLevel) { 
/// <summary>Enables Microsoft Ajax VSIntellisense extension logging</summary>
/// <param name="logLevel" type="String">Acceptable values are "info", "warning", and "error"</param>

    intellisense.logMessage("Passed in log level is " + logLevel);
    if(logLevel) {
        switch(logLevel.trim().toLowerCase())
        {
            case "info":
                intellisense["MicrosoftAjaxLogLevelInfo"] = true;
                intellisense.logMessage("Setting log level info.");
                // Intentional fallthrough
            case "warning":
                intellisense["MicrosoftAjaxLogLevelWarning"] = true;
                intellisense.logMessage("Setting log level warning.");
                // Intentional fallthrough
            case "error":
                intellisense["MicrosoftAjaxLogLevelError"] = true;
                intellisense.logMessage("Setting log level error.");
        }
    }
};


(function () {


    function logError(message) {
        intellisense.logMessage("MicrosoftAjax.intellisense error: " + message);
    }

    function logWarning(message) {
        intellisense.logMessage("MicrosoftAjax.intellisense warning: " + message);
    }

    function logInfo(message) {
        intellisense.logMessage("MicrosoftAjax.intellisense info: " + message);
    }

    // Override runtime validation functions to improve perf and avoid unnecessary errors
    Function._validateParams = function Function$_validateParams(params, expectedParams, validateParameterCount) {
        return null;
    }

    String._toFormattedString = function String$_toFormattedString(useLocale, args) {
        return "";
    }


    function getMetaFieldNames(functionObject) {
        /// <summary>Returns the names of the doc-comment fields defined in the passed in function</summary>
        /// <param name="functionObject" type="Function">Function to search for doc-comment field definitions</param>
        /// <returns type="Object">Map of meta-field names to "MetaField" marker string.</returns>

        if(!functionObject) {
            return null;
        }

        var functionDocComments = getFunctionDocComments(functionObject);
        if (!functionDocComments) {
            return null;
        }

        var fieldSet = {};

        var fieldNamesRegExp = /<field\s+[^>]*name=[\'\"]{1}([^\'\"]*)[\'\"]{1}/g;
        var fieldNameCaptureIndex = 1;
        var results;
        while (results = fieldNamesRegExp.exec(functionDocComments)) {
            var metaField = results[fieldNameCaptureIndex];
            if (logInfos) logInfo("Found meta-field: " + metaField); 
            fieldSet[metaField] = "MetaField";
        }

        return fieldSet;
    }

    function getFunctionDocComments(functionObject) {
        /// <summary>Returns XML doc-comment text of function</summary>
        /// <param name="functionObject" type="Function">Function to search for doc-comments</param>
        /// <returns type="String">Doc-comment content after stripping the leading ///</returns>

        var functionCode = functionObject.toString();
        var functionDocComments = "";

        var line;
        var nextLineStartPos = functionCode.indexOf('{');

        if (nextLineStartPos < 0)
            return null;
        else
            nextLineStartPos++;

        while ((line = getLine(nextLineStartPos, functionCode)) && (line.length > 0)) {
            var docCommentLine = getDocCommentLine(line);
            if (docCommentLine == null)
                break;
            else if (docCommentLine.length > 0)
                functionDocComments += docCommentLine;

            nextLineStartPos += line.length;
        }

        if (logInfos) logInfo("Got function doc-comments: " + functionDocComments);

        return functionDocComments;
    }

    function getLine(startPosition, multiLineText) {
        /// <summary>First line starting at the specified position within passed in text</summary>
        /// <param name="startPosition" type="Number">Search start position</param>
        /// <param name="multiLineText" type="String">Multi-line text to search for the next line</param>
        /// <returns type="String">Single line starting at the specified position or null at the end of the text.</returns>
   
        if (startPosition === undefined || multiLineText === undefined)
            return null;

        if(startPosition >= multiLineText.length)
            return null;

        for(var position = startPosition; position < multiLineText.length; position++) {
            if(multiLineText[position] == '\r' || multiLineText[position] == '\n') {
                if (multiLineText[position] == '\r' && position + 1 < multiLineText.length && multiLineText[position + 1] == '\n')
                    position++;
                return multiLineText.substring(startPosition, position + 1);
            }
        }

        return multiLineText.substring(startPosition);
    }

    function getDocCommentLine(line) {
        /// <summary>Returns text of the XML doc-comment one the line</summary>
        /// <param name="line" type="String"></param>
        /// <returns type="String">Doc-comment line content stripping leading ///, or null if the line is not a doc-comment line</returns>

        var docComment = "///";

        line = line.trim();

        // Blank lines are fine
        if (line.length == 0)
            return "";

        // Non-blank line that doesn't start with "///" is not considered a doc-comment line
        if (!line.startsWith(docComment))
            return null;
        
        // Don't allow more than three slashes at the start
        if (line.length > docComment.length && line[docComment.length + 1] == '/')
            return null;

        // Empty doc-comment lines are fine
        if (line.length == docComment.length)
            return "";

        return line.substring(docComment.length);
    }


    intellisense.addEventListener('statementcompletion', function (e) {

        function addInheritedItems() {
            
            if(intellisense["MicrosoftAjaxLogLevelInfo"]) logInfo("In addInheritedItems");

            if (contextMsAjaxType == "class") {
                if (e.target.resolveInheritance) {
                    var childMemberCache = new Object();
                    for (var member in e.target) {
                        childMemberCache[member] = member;
                    }
             
                    e.target.resolveInheritance();
             
                    for (var member in e.target) {
                        if (!childMemberCache[member]) {
                            var kind = isFunction(e.target[member]) ? "method" : "field";
                            e.items.push({ name: member, kind: kind, value: e.target[member], parentObject: e.target }); 
                        }
                    }
                }
            }

            if (logInfos) logInfo("Exiting addInheritedItems");
        }


        function isFunction(obj) {
            return Object.prototype.toString.call(obj) === "[object Function]";
        }

        function filterItems() {

            if (logInfos) logInfo("Entering filterItems for " + e.items.length + " items");

            e.items = e.items.filter(filterItem);

            if (logInfos) logInfo("Exiting filterItems");

        }

        function filterItem(item) {

            if (logInfos) logInfo("");
            if (logInfos) logInfo("Filtering item: " + item.name);

            var hidden = false;
            var msAjaxType = null;
        
            hidden = item.name[0] == "_";
            hidden |= item.name.indexOf("$") > 0;

            if (hidden) {
                var contextIsThis = false;
                if (e.targetName && e.targetName == "this") { 
                    contextIsThis = true;
                }
                if (logInfos) logInfo("Context is this: " + contextIsThis + " (it is " + e.targetName + ")");
                if (contextIsThis && !isGlobalScope) {
                    // Don"t hide any members when completion is invoked on "this."
                    // and "this" is not global scope
                    hidden = false;
                }
                else if (isGlobalScope) {
                    if (logInfos) logInfo("In global scope item value for item " + item.name + " is " + item.value);
                    msAjaxType = getMicrosoftAjaxTypeForObject(item.value);
                    if (msAjaxType == null) {
                        hidden = false;
                    }
                }
            }

            // For Enums and Flags, we show fields only
            if (!hidden && isEnumContext) {

                if (item.kind != "field") {
                    hidden = true;
                    if (logInfos) logInfo("Filtering out non-field member " + item.name + " on a " + contextMsAjaxType);
                }
                else if (enumMetaFields && (enumMetaFields[item.name] != "MetaField")) {
                    hidden = true;
                    if (logInfos) logInfo("Filtering out non-meta-field member " + item.name + " on a " + contextMsAjaxType);
                }
            }

            return !hidden;
        }

        function setItemsGlyphs() {
            e.items.forEach(setItemGlyph);
        }

        function setItemGlyph(item) {

            if (logInfos) logInfo("Setting item glyph for: " + item.name);

            if (isEnumContext) {
                item.glyph = "vs:GlyphGroupEnumMember";
            }
            else if (item.name.indexOf("set_") == 0 || item.name.indexOf("get_") == 0) {
                if (item.kind == "method") {
                    item.kind = "property";
                }
            }
            else if (item.name.indexOf("add_") == 0 || item.name.indexOf("remove_") == 0) {
                if (item.kind == "method") {
                    item.glyph = "vs:GlyphGroupEvent";
                }
            }
            else {
                var itemValue = item.value;
                if(itemValue) {
                    var ajaxType = getMicrosoftAjaxType(itemValue);
                    if (ajaxType != null) {
                        switch (ajaxType) {
                            case "class":
                                item.glyph = "vs:GlyphGroupClass";
                                break;
                            case "namespace":
                                item.glyph = "vs:GlyphGroupNamespace";
                                break;
                            case "interface":
                                item.glyph = "vs:GlyphGroupInterface";
                                break;
                            case "enum":
                            case "flags":
                                item.glyph = "vs:GlyphGroupEnum";
                            default:
                                if (logErrors) logError("Unknown Microsoft Ajax type: " + ajaxType);
                        }
                    }
                }
            }
        }

        function isFlagsOrEnumItem(item) {
            if (logInfos) logInfo("Entering isFlagsOrEnumItem for: " + item.name);
            var microsoftAjaxType = getMicrosoftAjaxTypeForObject(item.value);
            return isFlagsOrEnumType(microsoftAjaxType);
        }

        function isFlagsOrEnumType(microsoftAjaxType) {
            return microsoftAjaxType == "enum" || microsoftAjaxType == "flags";
        }

        function getMicrosoftAjaxTypeForObject(object) {
            
            var ajaxType = null;

            if (object) {
                ajaxType = getMicrosoftAjaxType(object);

                if (ajaxType == null) {
                    var objectConstructor = object.constructor;
                    if (objectConstructor) {
                        ajaxType = getMicrosoftAjaxType(objectConstructor);
                        if (ajaxType != null) { 
                            if (logInfos) logInfo("Got MS Ajax type for object constructor: " + ajaxType);
                        }
                    }
                }
                else {
                    if (logInfos) logInfo("Got MS Ajax type for object itself: " + ajaxType);
                }
            }

            return ajaxType;
        }

        function getMicrosoftAjaxType(obj) {

            var ajaxType = null;

            if (obj.__class)
                ajaxType = "class";
            else if (obj.__interface)
                ajaxType = "interface";
            else if (obj.__namespace)
                ajaxType = "namespace";
            else if (obj.__flags)
                ajaxType = "flags";
            else if (obj.__enum)
                ajaxType = "enum";

            return ajaxType;
        }

        function isGlobalScopeCompletionList() {

            var result = false;

            for (var i = 0; i < e.items.length; i++) {
                if (e.items[i].name == "break") {
                    result = true;
                    break;
                }
            }

            return result;
        }

        var logErrors = !!intellisense["MicrosoftAjaxLogLevelError"];
        var logWarnings = !!intellisense["MicrosoftAjaxLogLevelWarning"];
        var logInfos = !!intellisense["MicrosoftAjaxLogLevelInfo"];
        
        if (logInfos) logInfo("In addCompletionHandler\n");
//        if (logInfos) logInfo("e.target is " + e.target);
        if (logInfos) logInfo("e.targetName is " + e.targetName);
        if (logInfos) logInfo("e.offset is " + e.offset);

        var isGlobalScope = isGlobalScopeCompletionList();
        var contextMsAjaxType = isGlobalScope ? null : getMicrosoftAjaxTypeForObject(e.target);
        var isEnumContext = contextMsAjaxType != null ? isFlagsOrEnumType(contextMsAjaxType) : false;
        var enumMetaFields = isEnumContext ? getMetaFieldNames(e.target) : {};

        if (logInfos) logInfo("isGlobalScope is " + isGlobalScope);
        if (logInfos) logInfo("contextMsAjaxType is " + contextMsAjaxType + "\n");
        if (logInfos) logInfo("isEnumContext is " + isEnumContext + "\n");

        addInheritedItems();
        filterItems();
        setItemsGlyphs();
    });
    

    intellisense.addEventListener('signaturehelp', function (e) {

        function getPropertyType(comments) {
            /// <param name="comments" type="String" />

            var propertyType = "";

            var docCommentStart = comments.indexOf("<value");
            if (docCommentStart == -1) {
                if (logWarnings) logWarning("<value> doc comment is not found.");
                return "";
            }

            var docCommentEnd = comments.indexOf("</value>", docCommentStart);
            if (docCommentEnd == -1) {
                docCommentEnd = comments.indexOf("/>", docCommentStart);
                if (docCommentEnd == -1) {
                    if (logWarnings) logWarning("End of <value> doc comment is not found.");
                    return "";
                }
            }

            var typeStart = comments.indexOf("type", docCommentStart);
            if (typeStart == -1 || typeStart > docCommentEnd)
                return "";

            var typeValueStart = comments.indexOf("\"", typeStart);
            if (typeValueStart == -1 || typeValueStart > docCommentEnd)
                return "";

            var typeValueEnd = comments.indexOf("\"", typeValueStart + 1);
            if (typeValueEnd == -1 || typeValueEnd > docCommentEnd)
                return "";

            propertyType = comments.substring(typeValueStart + 1, typeValueEnd);

            return propertyType;
        }

        function fixupSetterSignature(signature) {
            if (logInfos) logInfo("Fixing signature for type " + getterType);
            if (getterType) {                
                if (signature.params.length > 0) {
                    signature.params[0].type = getterType;
                }
            }
            else {
                if(intellisense["MicrosoftAjaxLogLevelError"]) logError("No getter.");
            }
        }

        function fixupSetter() {

            var baseName = e.functionHelp.functionName.substring("set_".length);
            var getterName = "get_" + baseName;

            if (e.parentObject) {
                var getterFunction = e.parentObject[getterName];
                if (getterFunction) {
                    var comments = intellisense.getFunctionComments(getterFunction);
                    if (comments && comments.inside) { 
                        if (logInfos) logInfo("Comments is " + comments.inside);
                        getterType = getPropertyType(comments.inside);
                        if (logInfos) logInfo("Matching getter type is " + getterType);
                        if (e.functionHelp.signatures) {
                            e.functionHelp.signatures.forEach(fixupSetterSignature);
                        }
                    }
                    else { 
                        if (logWarnings) logWarning("No doc-comments in matching getter " + getterName);
                    }
                }
                else {
                    if (logWarnings) logWarning("No matching getter for seter " + e.functionHelp.functionName);
                }
            }
        }

        var logErrors = !!intellisense["MicrosoftAjaxLogLevelError"];
        var logWarnings = !!intellisense["MicrosoftAjaxLogLevelWarning"];
        var logInfos = !!intellisense["MicrosoftAjaxLogLevelInfo"];

        if (logInfos) logInfo("In addEventListener(functionHelp)");

        var getterType = null;

        if (e.functionHelp.functionName.indexOf("set_") == 0 && e.functionHelp.functionName.length > "set_".length) {
            fixupSetter();
        }
    });

})();



// SIG // Begin signature block
// SIG // MIIkHwYJKoZIhvcNAQcCoIIkEDCCJAwCAQExDzANBglg
// SIG // hkgBZQMEAgEFADB3BgorBgEEAYI3AgEEoGkwZzAyBgor
// SIG // BgEEAYI3AgEeMCQCAQEEEBDgyQbOONQRoqMAEEvTUJAC
// SIG // AQACAQACAQACAQACAQAwMTANBglghkgBZQMEAgEFAAQg
// SIG // Pn2USddfENSrBph1HxT6hUUW020zV31kQ0IMNBdJB/2g
// SIG // gg2TMIIGETCCA/mgAwIBAgITMwAAAI6HkaRXGl/KPgAA
// SIG // AAAAjjANBgkqhkiG9w0BAQsFADB+MQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQgQ29kZSBT
// SIG // aWduaW5nIFBDQSAyMDExMB4XDTE2MTExNzIyMDkyMVoX
// SIG // DTE4MDIxNzIyMDkyMVowgYMxCzAJBgNVBAYTAlVTMRMw
// SIG // EQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRt
// SIG // b25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRp
// SIG // b24xDTALBgNVBAsTBE1PUFIxHjAcBgNVBAMTFU1pY3Jv
// SIG // c29mdCBDb3Jwb3JhdGlvbjCCASIwDQYJKoZIhvcNAQEB
// SIG // BQADggEPADCCAQoCggEBANCH1EIrfp3ZxnrUosPjFZLS
// SIG // U52VF8lSNvpUv6sQr+nJ58wmU8PCc79t1gDlANzpamc0
// SIG // MPYWF7QBpZV8i7mkLOaLO3n2Iwx5j/NS30ABHMLGA53r
// SIG // Wc9z6dhxOZvwziVZLdLJWwrvftYyDl10EgTsngRTpmsC
// SIG // Z/hNWYt34Csh4O/ApEUSzwN7A8Y5w9Qi3FVcd0L/nLLl
// SIG // VWdoui12an9mU0fVRwrMON6Ne5cZfYLQJviljuWh8F5k
// SIG // EOT56yfG8uAI0A3yZ8DY8i/7idoV+a4PPgCXB9ELPnDU
// SIG // d6tyeEGYB7gXzKKxX+y981Bno9eU8NKLVY9TppWT5rJm
// SIG // z8k3aORjx88CAwEAAaOCAYAwggF8MB8GA1UdJQQYMBYG
// SIG // CisGAQQBgjdMCAEGCCsGAQUFBwMDMB0GA1UdDgQWBBSr
// SIG // yNbtshXSqo7xzO1sOPdFStCKuzBSBgNVHREESzBJpEcw
// SIG // RTENMAsGA1UECxMETU9QUjE0MDIGA1UEBRMrMjMwMDEy
// SIG // K2IwNTBjNmU3LTc2NDEtNDQxZi1iYzRhLTQzNDgxZTQx
// SIG // NWQwODAfBgNVHSMEGDAWgBRIbmTlUAXTgqoXNzcitW2o
// SIG // ynUClTBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8vd3d3
// SIG // Lm1pY3Jvc29mdC5jb20vcGtpb3BzL2NybC9NaWNDb2RT
// SIG // aWdQQ0EyMDExXzIwMTEtMDctMDguY3JsMGEGCCsGAQUF
// SIG // BwEBBFUwUzBRBggrBgEFBQcwAoZFaHR0cDovL3d3dy5t
// SIG // aWNyb3NvZnQuY29tL3BraW9wcy9jZXJ0cy9NaWNDb2RT
// SIG // aWdQQ0EyMDExXzIwMTEtMDctMDguY3J0MAwGA1UdEwEB
// SIG // /wQCMAAwDQYJKoZIhvcNAQELBQADggIBAESJAqxpU/PE
// SIG // trvUjGBT58psqElpZr6lmkGZOtid0lcCUWr6v5uW26Ym
// SIG // fQlW6NztJXV6pUdSqB5LFlPz7g+awwSVKcGChKRWMfyg
// SIG // ipGVtb9azqkBH2RGoebK8dd0e7+SCFFefDMCXlE7m+XY
// SIG // Ll8CTAmcGkPace3k2eei2nQsF63lDLUY9VQJ1L4cc80g
// SIG // e6T6yNvY2zqu+pDFo72VZa5GLVcpWNaS8GzaY/GPM6J+
// SIG // OHZe3fM17ayaO2KB0E4ZfEh8sAuPOMwtvNU5ZamVwQPi
// SIG // ksm5q9JXCqrcUgsuViej4piXV468qVluJJKOguIJc4LZ
// SIG // NYPMn3/RBI6IuOKag1iw1JrmMfqUR459puJOefPY02oz
// SIG // FlBw8UK7mAnp/8yVVVsIv5JSqAjE8ejx/0DX+Zo2nf26
// SIG // kIXSVT5QrUYf7yUMuJ46SARj73iYol0DDQLY3CCr5la1
// SIG // 3u8WZsPXVYIeT4J4yZ5UGhBgtxerQBORrrAZwZozne4y
// SIG // cs1lzE9GmC0PUWAefPv+2+gHeQf3oTM4/gma2497tjq9
// SIG // hYa4zLx9ATC3ex2pXRu9zE0X925HM9VA32rKLlG4tbnP
// SIG // wwTTO+Xj6RCM66e63qQuM2opLxRK6h7BIjg1BYXvwgQA
// SIG // DWvB2JYUSBWvflKwuGDEUrVKgreFKgBJKiaDJ1pB3r3V
// SIG // Zkm8C5x4cAm8MIIHejCCBWKgAwIBAgIKYQ6Q0gAAAAAA
// SIG // AzANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UEBhMCVVMx
// SIG // EzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNVBAcTB1Jl
// SIG // ZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBDb3Jwb3Jh
// SIG // dGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJvb3QgQ2Vy
// SIG // dGlmaWNhdGUgQXV0aG9yaXR5IDIwMTEwHhcNMTEwNzA4
// SIG // MjA1OTA5WhcNMjYwNzA4MjEwOTA5WjB+MQswCQYDVQQG
// SIG // EwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UE
// SIG // BxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENv
// SIG // cnBvcmF0aW9uMSgwJgYDVQQDEx9NaWNyb3NvZnQgQ29k
// SIG // ZSBTaWduaW5nIFBDQSAyMDExMIICIjANBgkqhkiG9w0B
// SIG // AQEFAAOCAg8AMIICCgKCAgEAq/D6chAcLq3YbqqCEE00
// SIG // uvK2WCGfQhsqa+laUKq4BjgaBEm6f8MMHt03a8YS2Avw
// SIG // OMKZBrDIOdUBFDFC04kNeWSHfpRgJGyvnkmc6Whe0t+b
// SIG // U7IKLMOv2akrrnoJr9eWWcpgGgXpZnboMlImEi/nqwhQ
// SIG // z7NEt13YxC4Ddato88tt8zpcoRb0RrrgOGSsbmQ1eKag
// SIG // Yw8t00CT+OPeBw3VXHmlSSnnDb6gE3e+lD3v++MrWhAf
// SIG // TVYoonpy4BI6t0le2O3tQ5GD2Xuye4Yb2T6xjF3oiU+E
// SIG // GvKhL1nkkDstrjNYxbc+/jLTswM9sbKvkjh+0p2ALPVO
// SIG // VpEhNSXDOW5kf1O6nA+tGSOEy/S6A4aN91/w0FK/jJSH
// SIG // vMAhdCVfGCi2zCcoOCWYOUo2z3yxkq4cI6epZuxhH2rh
// SIG // KEmdX4jiJV3TIUs+UsS1Vz8kA/DRelsv1SPjcF0PUUZ3
// SIG // s/gA4bysAoJf28AVs70b1FVL5zmhD+kjSbwYuER8ReTB
// SIG // w3J64HLnJN+/RpnF78IcV9uDjexNSTCnq47f7Fufr/zd
// SIG // sGbiwZeBe+3W7UvnSSmnEyimp31ngOaKYnhfsi+E11ec
// SIG // XL93KCjx7W3DKI8sj0A3T8HhhUSJxAlMxdSlQy90lfdu
// SIG // +HggWCwTXWCVmj5PM4TasIgX3p5O9JawvEagbJjS4NaI
// SIG // jAsCAwEAAaOCAe0wggHpMBAGCSsGAQQBgjcVAQQDAgEA
// SIG // MB0GA1UdDgQWBBRIbmTlUAXTgqoXNzcitW2oynUClTAZ
// SIG // BgkrBgEEAYI3FAIEDB4KAFMAdQBiAEMAQTALBgNVHQ8E
// SIG // BAMCAYYwDwYDVR0TAQH/BAUwAwEB/zAfBgNVHSMEGDAW
// SIG // gBRyLToCMZBDuRQFTuHqp8cx0SOJNDBaBgNVHR8EUzBR
// SIG // ME+gTaBLhklodHRwOi8vY3JsLm1pY3Jvc29mdC5jb20v
// SIG // cGtpL2NybC9wcm9kdWN0cy9NaWNSb29DZXJBdXQyMDEx
// SIG // XzIwMTFfMDNfMjIuY3JsMF4GCCsGAQUFBwEBBFIwUDBO
// SIG // BggrBgEFBQcwAoZCaHR0cDovL3d3dy5taWNyb3NvZnQu
// SIG // Y29tL3BraS9jZXJ0cy9NaWNSb29DZXJBdXQyMDExXzIw
// SIG // MTFfMDNfMjIuY3J0MIGfBgNVHSAEgZcwgZQwgZEGCSsG
// SIG // AQQBgjcuAzCBgzA/BggrBgEFBQcCARYzaHR0cDovL3d3
// SIG // dy5taWNyb3NvZnQuY29tL3BraW9wcy9kb2NzL3ByaW1h
// SIG // cnljcHMuaHRtMEAGCCsGAQUFBwICMDQeMiAdAEwAZQBn
// SIG // AGEAbABfAHAAbwBsAGkAYwB5AF8AcwB0AGEAdABlAG0A
// SIG // ZQBuAHQALiAdMA0GCSqGSIb3DQEBCwUAA4ICAQBn8oal
// SIG // mOBUeRou09h0ZyKbC5YR4WOSmUKWfdJ5DJDBZV8uLD74
// SIG // w3LRbYP+vj/oCso7v0epo/Np22O/IjWll11lhJB9i0ZQ
// SIG // VdgMknzSGksc8zxCi1LQsP1r4z4HLimb5j0bpdS1HXeU
// SIG // OeLpZMlEPXh6I/MTfaaQdION9MsmAkYqwooQu6SpBQyb
// SIG // 7Wj6aC6VoCo/KmtYSWMfCWluWpiW5IP0wI/zRive/DvQ
// SIG // vTXvbiWu5a8n7dDd8w6vmSiXmE0OPQvyCInWH8MyGOLw
// SIG // xS3OW560STkKxgrCxq2u5bLZ2xWIUUVYODJxJxp/sfQn
// SIG // +N4sOiBpmLJZiWhub6e3dMNABQamASooPoI/E01mC8Cz
// SIG // TfXhj38cbxV9Rad25UAqZaPDXVJihsMdYzaXht/a8/jy
// SIG // FqGaJ+HNpZfQ7l1jQeNbB5yHPgZ3BtEGsXUfFL5hYbXw
// SIG // 3MYbBL7fQccOKO7eZS/sl/ahXJbYANahRr1Z85elCUtI
// SIG // EJmAH9AAKcWxm6U/RXceNcbSoqKfenoi+kiVH6v7RyOA
// SIG // 9Z74v2u3S5fi63V4GuzqN5l5GEv/1rMjaHXmr/r8i+sL
// SIG // gOppO6/8MO0ETI7f33VtY5E90Z1WTk+/gFcioXgRMiF6
// SIG // 70EKsT/7qMykXcGhiJtXcVZOSEXAQsmbdlsKgEhr/Xmf
// SIG // wb1tbWrJUnMTDXpQzTGCFeQwghXgAgEBMIGVMH4xCzAJ
// SIG // BgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAw
// SIG // DgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3Nv
// SIG // ZnQgQ29ycG9yYXRpb24xKDAmBgNVBAMTH01pY3Jvc29m
// SIG // dCBDb2RlIFNpZ25pbmcgUENBIDIwMTECEzMAAACOh5Gk
// SIG // Vxpfyj4AAAAAAI4wDQYJYIZIAWUDBAIBBQCggdIwGQYJ
// SIG // KoZIhvcNAQkDMQwGCisGAQQBgjcCAQQwHAYKKwYBBAGC
// SIG // NwIBCzEOMAwGCisGAQQBgjcCARUwLwYJKoZIhvcNAQkE
// SIG // MSIEIJ8NTHzwtBz6DImgRYg6wYt5FcBv+FTGah+LQ84I
// SIG // +hoBMGYGCisGAQQBgjcCAQwxWDBWoDyAOgBNAGkAYwBy
// SIG // AG8AcwBvAGYAdABBAGoAYQB4AC4AaQBuAHQAZQBsAGwA
// SIG // aQBzAGUAbgBzAGUALgBqAHOhFoAUaHR0cDovL21pY3Jv
// SIG // c29mdC5jb20wDQYJKoZIhvcNAQEBBQAEggEAZBV+Tkr6
// SIG // xNsVlCMzGMypcj7bq+rvxifncp9VA74nZL5PBpcVQZZw
// SIG // iDGSnJQLlOto4FBd/8aaLCrBIod8eFQnlWm4J/vTuREA
// SIG // mIBngoMpa56veUxGWOMPaxzvx93ELMpz/3MLBtXDB+qA
// SIG // 2sjI8OVuGWMUxuQK+bFxteyi0wAB2NSPj/2nVGWhGkS5
// SIG // mRXsl10JU/exrrKFudwIndiX6Yrq0N+zZ1NzV4/BGNs7
// SIG // bvhcScYbiNjUPjwIMANTtouqWXZY8Arr1Ftr6jcn3gE5
// SIG // ox341BIqihsQFWYThqu7r+Y8SvYSHxIBDD4vy5D6RThz
// SIG // Hus4d7bU5YCZ4ShiUdy57llmdaGCE0owghNGBgorBgEE
// SIG // AYI3AwMBMYITNjCCEzIGCSqGSIb3DQEHAqCCEyMwghMf
// SIG // AgEDMQ8wDQYJYIZIAWUDBAIBBQAwggE9BgsqhkiG9w0B
// SIG // CRABBKCCASwEggEoMIIBJAIBAQYKKwYBBAGEWQoDATAx
// SIG // MA0GCWCGSAFlAwQCAQUABCA/hI9CPQJThBS1FBVdYQvI
// SIG // M3QGrao2QgpsZ+2GdG7iuQIGWIurMiltGBMyMDE3MDIw
// SIG // OTAzNDMzMi4yMTRaMAcCAQGAAgH0oIG5pIG2MIGzMQsw
// SIG // CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQ
// SIG // MA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9z
// SIG // b2Z0IENvcnBvcmF0aW9uMQ0wCwYDVQQLEwRNT1BSMScw
// SIG // JQYDVQQLEx5uQ2lwaGVyIERTRSBFU046RjUyOC0zNzc3
// SIG // LThBNzYxJTAjBgNVBAMTHE1pY3Jvc29mdCBUaW1lLVN0
// SIG // YW1wIFNlcnZpY2Wggg7NMIIGcTCCBFmgAwIBAgIKYQmB
// SIG // KgAAAAAAAjANBgkqhkiG9w0BAQsFADCBiDELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEyMDAGA1UEAxMpTWljcm9zb2Z0IFJv
// SIG // b3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5IDIwMTAwHhcN
// SIG // MTAwNzAxMjEzNjU1WhcNMjUwNzAxMjE0NjU1WjB8MQsw
// SIG // CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQ
// SIG // MA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9z
// SIG // b2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1NaWNyb3Nv
// SIG // ZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDCCASIwDQYJKoZI
// SIG // hvcNAQEBBQADggEPADCCAQoCggEBAKkdDbx3EYo6IOz8
// SIG // E5f1+n9plGt0VBDVpQoAgoX77XxoSyxfxcPlYcJ2tz5m
// SIG // K1vwFVMnBDEfQRsalR3OCROOfGEwWbEwRA/xYIiEVEMM
// SIG // 1024OAizQt2TrNZzMFcmgqNFDdDq9UeBzb8kYDJYYEby
// SIG // WEeGMoQedGFnkV+BVLHPk0ySwcSmXdFhE24oxhr5hoC7
// SIG // 32H8RsEnHSRnEnIaIYqvS2SJUGKxXf13Hz3wV3WsvYpC
// SIG // TUBR0Q+cBj5nf/VmwAOWRH7v0Ev9buWayrGo8noqCjHw
// SIG // 2k4GkbaICDXoeByw6ZnNPOcvRLqn9NxkvaQBwSAJk3jN
// SIG // /LzAyURdXhacAQVPIk0CAwEAAaOCAeYwggHiMBAGCSsG
// SIG // AQQBgjcVAQQDAgEAMB0GA1UdDgQWBBTVYzpcijGQ80N7
// SIG // fEYbxTNoWoVtVTAZBgkrBgEEAYI3FAIEDB4KAFMAdQBi
// SIG // AEMAQTALBgNVHQ8EBAMCAYYwDwYDVR0TAQH/BAUwAwEB
// SIG // /zAfBgNVHSMEGDAWgBTV9lbLj+iiXGJo0T2UkFvXzpoY
// SIG // xDBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWNS
// SIG // b29DZXJBdXRfMjAxMC0wNi0yMy5jcmwwWgYIKwYBBQUH
// SIG // AQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY1Jvb0NlckF1
// SIG // dF8yMDEwLTA2LTIzLmNydDCBoAYDVR0gAQH/BIGVMIGS
// SIG // MIGPBgkrBgEEAYI3LgMwgYEwPQYIKwYBBQUHAgEWMWh0
// SIG // dHA6Ly93d3cubWljcm9zb2Z0LmNvbS9QS0kvZG9jcy9D
// SIG // UFMvZGVmYXVsdC5odG0wQAYIKwYBBQUHAgIwNB4yIB0A
// SIG // TABlAGcAYQBsAF8AUABvAGwAaQBjAHkAXwBTAHQAYQB0
// SIG // AGUAbQBlAG4AdAAuIB0wDQYJKoZIhvcNAQELBQADggIB
// SIG // AAfmiFEN4sbgmD+BcQM9naOhIW+z66bM9TG+zwXiqf76
// SIG // V20ZMLPCxWbJat/15/B4vceoniXj+bzta1RXCCtRgkQS
// SIG // +7lTjMz0YBKKdsxAQEGb3FwX/1z5Xhc1mCRWS3TvQhDI
// SIG // r79/xn/yN31aPxzymXlKkVIArzgPF/UveYFl2am1a+TH
// SIG // zvbKegBvSzBEJCI8z+0DpZaPWSm8tv0E4XCfMkon/VWv
// SIG // L/625Y4zu2JfmttXQOnxzplmkIz/amJ/3cVKC5Em4jns
// SIG // GUpxY517IW3DnKOiPPp/fZZqkHimbdLhnPkd/DjYlPTG
// SIG // pQqWhqS9nhquBEKDuLWAmyI4ILUl5WTs9/S/fmNZJQ96
// SIG // LjlXdqJxqgaKD4kWumGnEcua2A5HmoDF0M2n0O99g/Dh
// SIG // O3EJ3110mCIIYdqwUB5vvfHhAN/nMQekkzr3ZUd46Pio
// SIG // SKv33nJ+YWtvd6mBy6cJrDm77MbL2IK0cs0d9LiFAR6A
// SIG // +xuJKlQ5slvayA1VmXqHczsI5pgt6o3gMy4SKfXAL1Qn
// SIG // IffIrE7aKLixqduWsqdCosnPGUFN4Ib5KpqjEWYw07t0
// SIG // MkvfY3v1mYovG8chr1m1rtxEPJdQcdeh0sVV42neV8HR
// SIG // 3jDA/czmTfsNv11P6Z0eGTgvvM9YBS7vDaBQNdrvCScc
// SIG // 1bN+NR4Iuto229Nfj950iEkSMIIE2jCCA8KgAwIBAgIT
// SIG // MwAAALCG6ZIgCl3q+AAAAAAAsDANBgkqhkiG9w0BAQsF
// SIG // ADB8MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGlu
// SIG // Z3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMV
// SIG // TWljcm9zb2Z0IENvcnBvcmF0aW9uMSYwJAYDVQQDEx1N
// SIG // aWNyb3NvZnQgVGltZS1TdGFtcCBQQ0EgMjAxMDAeFw0x
// SIG // NjA5MDcxNzU2NTZaFw0xODA5MDcxNzU2NTZaMIGzMQsw
// SIG // CQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQ
// SIG // MA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9z
// SIG // b2Z0IENvcnBvcmF0aW9uMQ0wCwYDVQQLEwRNT1BSMScw
// SIG // JQYDVQQLEx5uQ2lwaGVyIERTRSBFU046RjUyOC0zNzc3
// SIG // LThBNzYxJTAjBgNVBAMTHE1pY3Jvc29mdCBUaW1lLVN0
// SIG // YW1wIFNlcnZpY2UwggEiMA0GCSqGSIb3DQEBAQUAA4IB
// SIG // DwAwggEKAoIBAQDw5fCNlFmpmtdsCQd3FCFZXbc9eLg1
// SIG // rfUwMf/O4f/W6RrJg5gj+5AQwZLsOrxQbJC9XPFrrUyi
// SIG // 9WGlh+EprKM8Et9/xACCzr20Cl/LuduatxktWu0HAK1U
// SIG // /TOs9vgSJEokZ1fauEuhrA+A+Tm9IA21p8QsS/GhVuby
// SIG // Lye5JsEzJdkrDDByUIRrkmqVjPL6CE24LiTVQ9Pc6/N0
// SIG // aoizybRg3MllrV8J5RFqFDTB5FcGEkbmoL2EWiRCQ/a8
// SIG // 9CxVmVqNs4imqhKUIr6GtUqJjKpHsKDFHxuPnPBibVSd
// SIG // MtOpxJtT6blyO78Xnq9YXJ3GK1Ahu9iWzDbvjaZz2a27
// SIG // Q3AVAgMBAAGjggEbMIIBFzAdBgNVHQ4EFgQU/KgHUtnv
// SIG // Kf6YQzwVXHRet39z4K8wHwYDVR0jBBgwFoAU1WM6XIox
// SIG // kPNDe3xGG8UzaFqFbVUwVgYDVR0fBE8wTTBLoEmgR4ZF
// SIG // aHR0cDovL2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwv
// SIG // cHJvZHVjdHMvTWljVGltU3RhUENBXzIwMTAtMDctMDEu
// SIG // Y3JsMFoGCCsGAQUFBwEBBE4wTDBKBggrBgEFBQcwAoY+
// SIG // aHR0cDovL3d3dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0
// SIG // cy9NaWNUaW1TdGFQQ0FfMjAxMC0wNy0wMS5jcnQwDAYD
// SIG // VR0TAQH/BAIwADATBgNVHSUEDDAKBggrBgEFBQcDCDAN
// SIG // BgkqhkiG9w0BAQsFAAOCAQEAkv2A93W9ZZA+F83VFYPF
// SIG // jgKRO6xOfWDvjzkGk5DjD6pFPYk/Av3sb7hQkAlshNI3
// SIG // IZmxwYZ2HeQNxo7/GOCi+ka1hXd0bk4MREXQvNK2BH5w
// SIG // Sw/WqwdpVkp2ZOj5qkejo4bc9M9EuEkQW2eP0dp5rjrd
// SIG // h1MG6I9q/H/X5KOGRRUNkWIiOpBK49hoAUnJLQ5reGwR
// SIG // AvSPTRFgc6gDIQ2X4w9ydbv96A646/wgQZ2Ok/3FM3M+
// SIG // OXq9ajQeOUdiEbUc71f0c4Nxn6gUZb7kA45NbcQBMxt+
// SIG // V+yh8xyXqTin9Kg6OfmJNfxdoyKuCr2NDKsxEm7pvWEW
// SIG // 7PQZOiSFYl+psqGCA3YwggJeAgEBMIHjoYG5pIG2MIGz
// SIG // MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3Rv
// SIG // bjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWlj
// SIG // cm9zb2Z0IENvcnBvcmF0aW9uMQ0wCwYDVQQLEwRNT1BS
// SIG // MScwJQYDVQQLEx5uQ2lwaGVyIERTRSBFU046RjUyOC0z
// SIG // Nzc3LThBNzYxJTAjBgNVBAMTHE1pY3Jvc29mdCBUaW1l
// SIG // LVN0YW1wIFNlcnZpY2WiJQoBATAJBgUrDgMCGgUAAxUA
// SIG // vIT7nVsS2sc2hTuIZp6jFhjVzByggcIwgb+kgbwwgbkx
// SIG // CzAJBgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9u
// SIG // MRAwDgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNy
// SIG // b3NvZnQgQ29ycG9yYXRpb24xDTALBgNVBAsTBE1PUFIx
// SIG // JzAlBgNVBAsTHm5DaXBoZXIgTlRTIEVTTjo1N0Y2LUMx
// SIG // RTAtNTU0QzErMCkGA1UEAxMiTWljcm9zb2Z0IFRpbWUg
// SIG // U291cmNlIE1hc3RlciBDbG9jazANBgkqhkiG9w0BAQUF
// SIG // AAIFANxGNIgwIhgPMjAxNzAyMDkwMDIxMjhaGA8yMDE3
// SIG // MDIxMDAwMjEyOFowdDA6BgorBgEEAYRZCgQBMSwwKjAK
// SIG // AgUA3EY0iAIBADAHAgEAAgIBxjAHAgEAAgIbMDAKAgUA
// SIG // 3EeGCAIBADA2BgorBgEEAYRZCgQCMSgwJjAMBgorBgEE
// SIG // AYRZCgMBoAowCAIBAAIDFuNgoQowCAIBAAIDB6EgMA0G
// SIG // CSqGSIb3DQEBBQUAA4IBAQAYjBhhG6oLiiNv6mZogqxu
// SIG // koDvyR7/qklj9SnQgR3cq3XaBEnK5RUjqFylWQ4hdcz+
// SIG // VtdfPhwDB95pqsLxwkSuqt7PP57+qMwQxp7GzGkYz5Yw
// SIG // w9QXfepNIlQQYkQV+TgoJ721V0d/lIvcg+eSIqN0BciX
// SIG // CIEhMfvDXqtipg31ftSdx4yO02s4eMTVhaIzz4PdSH0a
// SIG // /6iS3fIdXsPDyqjJZ3ICGYK5lV3irizgVWli51yKkzN8
// SIG // 9NnIw5U1fPhscCNu+D8WPH+vXgq96Qe3BuyvdwiJE7cg
// SIG // bNafH2P8nJUoXxGb/l/Xn/HgGw9po4M5h/GyFY32z1CP
// SIG // VM+mQ3HLA+dnMYIC9TCCAvECAQEwgZMwfDELMAkGA1UE
// SIG // BhMCVVMxEzARBgNVBAgTCldhc2hpbmd0b24xEDAOBgNV
// SIG // BAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1pY3Jvc29mdCBD
// SIG // b3Jwb3JhdGlvbjEmMCQGA1UEAxMdTWljcm9zb2Z0IFRp
// SIG // bWUtU3RhbXAgUENBIDIwMTACEzMAAACwhumSIApd6vgA
// SIG // AAAAALAwDQYJYIZIAWUDBAIBBQCgggEyMBoGCSqGSIb3
// SIG // DQEJAzENBgsqhkiG9w0BCRABBDAvBgkqhkiG9w0BCQQx
// SIG // IgQgIVAb3HNieyGozaoExO98blxFiO9iLqhNpflzEylF
// SIG // r28wgeIGCyqGSIb3DQEJEAIMMYHSMIHPMIHMMIGxBBS8
// SIG // hPudWxLaxzaFO4hmnqMWGNXMHDCBmDCBgKR+MHwxCzAJ
// SIG // BgNVBAYTAlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAw
// SIG // DgYDVQQHEwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3Nv
// SIG // ZnQgQ29ycG9yYXRpb24xJjAkBgNVBAMTHU1pY3Jvc29m
// SIG // dCBUaW1lLVN0YW1wIFBDQSAyMDEwAhMzAAAAsIbpkiAK
// SIG // Xer4AAAAAACwMBYEFFHUlpu9yIf7hQbEOljWLP8rKJ7c
// SIG // MA0GCSqGSIb3DQEBCwUABIIBAOFa+yEVtvviylJZmuWd
// SIG // fHUsVO3fqRrUEjJUIV1+ZdpXxuKkgSx9hdk8ViyQ8kQh
// SIG // MvchTFmuvv4fn+jEcHim13KCClT+10U66/VejiMpC2pt
// SIG // nvK+tyHD75ekhtkN/8xJBlZfE6n5N6ZGve0+oRSCoOXl
// SIG // UPRqNv6QpPU5Q2kOgNdb2GYCEIDapDEcpp30FIX5jbq/
// SIG // PdFzu+dFJoPC5KQW8lx3TowgUrKvLi1bb9wDY2GMO3cC
// SIG // 4+YhDsdRZ15f6LbgWu9VBvR9w89mcpTKGypbJ3KpCLw2
// SIG // Hm1HvZnileoHq39a4mvG2OuUB/46q+NVLyngWJcrQk1a
// SIG // l2M/FzQcScoSMM4=
// SIG // End signature block
